#include <LSM303AGR_ACC_Sensor.h>
#include <LSM303AGR_MAG_Sensor.h>
#include <Wire.h>

// Define pins for hardware serial connections
#define GSM_TX_PIN 37 // TX pin for GSM module
#define GSM_RX_PIN 38 // RX pin for GSM module
#define GPS_TX_PIN 43 // TX pin for GPS module
#define GPS_RX_PIN 44 // RX pin for GPS module

// Define other pins
#define BUZZER_PIN 36       // Buzzer pin
#define SOS_BUTTON_PIN 7    // SOS button pin

// Define thresholds
#define FALL_THRESHOLD 15.0 // Threshold for detecting a fall
#define TEMP_THRESHOLD 38.0 // Threshold for high temperature (Celsius)

// Create objects
LSM303AGR_ACC_Sensor Acc(&Wire);
LSM303AGR_MAG_Sensor Mag(&Wire);

// Use hardware serial
HardwareSerial GSM(1); // HardwareSerial 1 for GSM
HardwareSerial GPS(2); // HardwareSerial 2 for GPS

float acceleration_x, acceleration_y, acceleration_z;
float temperature;

void setup() {
  Serial.begin(115200);  // Debugging serial port
  GSM.begin(9600, SERIAL_8N1, GSM_RX_PIN, GSM_TX_PIN); // Initialize GSM module
  GPS.begin(9600, SERIAL_8N1, GPS_RX_PIN, GPS_TX_PIN); // Initialize GPS module

  // Initialize pins
  pinMode(BUZZER_PIN, OUTPUT);
  pinMode(SOS_BUTTON_PIN, INPUT_PULLUP);

  digitalWrite(BUZZER_PIN, LOW);

  // Initialize accelerometer and temperature sensor
  Wire.begin();
  Acc.begin();
  Acc.Enable();
  Acc.EnableTemperatureSensor();
  Mag.begin();
  Mag.Enable();

  Serial.println("System Initialized");
}

void loop() {
  // Read accelerometer values
  Acc.Get_X_Axis_Acceleration(&acceleration_x);
  Acc.Get_Y_Axis_Acceleration(&acceleration_y);
  Acc.Get_Z_Axis_Acceleration(&acceleration_z);

  // Read temperature
  Acc.GetTemperature(&temperature);
  Serial.print("Temperature: ");
  Serial.print(temperature, 2);
  Serial.println(" °C");

  // Monitor SOS button
  if (digitalRead(SOS_BUTTON_PIN) == LOW) {
    Serial.println("SOS Button Pressed!");
    triggerSOS();
  }

  // Detect sudden changes in acceleration (fall detection)
  if (detectFall()) {
    Serial.println("Fall Detected!");
    triggerSOS();
  }

  // Check for high temperature
  if (temperature >= TEMP_THRESHOLD) {
    Serial.println("High Temperature Detected!");
    triggerSOS();
  }

  delay(100); // Delay for stability
}

bool detectFall() {
  // Calculate total acceleration vector
  float total_acceleration = sqrt(
      acceleration_x * acceleration_x +
      acceleration_y * acceleration_y +
      acceleration_z * acceleration_z);

  Serial.print("Total Acceleration: ");
  Serial.println(total_acceleration, 2);

  // Check if the total acceleration exceeds the fall threshold
  return total_acceleration >= FALL_THRESHOLD;
}

void triggerSOS() {
  digitalWrite(BUZZER_PIN, HIGH);
  delay(1000); // Let buzzer ring for 1 second
  digitalWrite(BUZZER_PIN, LOW);

  // Get GPS location
  String location = getGPSLocation();

  // Send SOS message
  sendSMS("SOS ALERT! Emergency Detected! Location: " + location);

  delay(2000); // Prevent spamming
}

String getGPSLocation() {
  String location = "Not Available";

  Serial.println("Fetching GPS Location...");
  while (GPS.available()) {
    char c = GPS.read();
    Serial.print(c);

    // Parse GPS data
    if (c == '\n' && GPS.find("GPRMC")) {
      float latitude = GPS.parseFloat();
      float longitude = GPS.parseFloat();

      location = String(latitude, 6) + ", " + String(longitude, 6);
      break;
    }
  }

  return location;
}

void sendSMS(String message) {
  GSM.println("AT"); // Handshake with GSM module
  delay(1000);
  GSM.println("AT+CMGF=1"); // Set SMS mode
  delay(1000);
  GSM.println("AT+CMGS=\"+916304434642\""); // Replace with a valid phone number
  delay(1000);
  GSM.print(message);
  GSM.write(26); // End SMS
  delay(1000);

  Serial.println("SMS Sent: " + message);
}
